<?php
/**
 * Load processors
 *
 * @package Caldera_Forms
 * @author    Josh Pollock <Josh@CalderaWP.com>
 * @license   GPL-2.0+
 * @link
 * @copyright 2015 CalderaWP LLC
 */
class Caldera_Forms_Processor_Load {

	/**
	 * Holds validated processors.
	 *
	 * @since 1.3.0
	 *
	 * @access private
	 *
	 * @var array
	 */
	private $processors = array();

	/**
	 * Holds accumulated errors notices.
	 *
	 * @since 1.3.0
	 *
	 * @access private
	 *
	 * @var array
	 */
	private $notices = array();

	/**
	 * Holds class instance
	 *
	 * @since 1.3.0
	 *
	 * @access private
	 *
	 * @var Caldera_Forms_Processor_Load
	 */
	private static $init;

	/**
	 * Constructor for class
	 *
	 * @since 1.3.0
	 *
	 * @access private
	 *
	 * @var array
	 */
	private function __construct() {
		$this->load_processors();

	}

	/**
	 * Get all valid processors
	 *
	 * @since 1.3.0
	 *
	 * @return array
	 */
	public function get_processors() {
		return $this->processors;

	}

	/**
	 * Get any errors that were generated by invalid processors.
	 *
	 * @since 1.3.0
	 *
	 * @return array
	 */
	public function get_notices() {
		return $this->notices;

	}

	/**
	 * Validate and load processors
	 *
	 * @since 1.3.0
	 *
	 * @access private
	 */
	private function load_processors() {
		/**
		 * Runs before processors are loaded
		 *
		 * Uses "caldera_forms_get_form_processors" filter to add processors
		 *
		 * @since 1.3.5.3
		 */
		do_action( 'caldera_forms_pre_load_processors' );

		/**
		 * Add processors
		 *
		 * @since 1.3.0
		 *
		 * @param array $processors
		 */
		$processors = apply_filters( 'caldera_forms_get_form_processors', array() );

		$this->processors = array();
		if ( is_array( $processors ) && ! empty( $processors ) ){
			foreach ( $processors as $id => $processor ) {
				$processor = $this->validate_processor( $processor );
				if ( $processor ) {
					$this->processors[ $id ] = $processor;
				}

			}
		}

		/**
		 * Runs after processors are loaded
		 *
		 * Use: $processors = Caldera_Forms_Processor_Load::get_instance()->get_processors(); to get processors
		 *
		 * @since 1.3.5.3
		 */
		do_action( 'caldera_forms_post_load_processors' );


	}

	/**
	 * Validates a processor.
	 *
	 * @param $processor
	 *
	 * @since 1.3.0
	 *
	 * @access private
	 *
	 * @return bool|array Processor config array, or false if invalid.
	 */
	private function validate_processor( $processor ) {
		$dependencies = array(
			'php_ver' => false,
			'cf_ver' => false,
			'wp_ver' => false,
		);

		//backwards compat, woo!
		$processor = wp_parse_args( $processor, $dependencies );

		$fail = false;
		$pattern = '%1 requires %2s version %3s or later. Current version is %4s.';
		if ( false != $processor[ 'cf_ver' ] ) {
			if ( ! version_compare( CFCORE_VER, $processor[ 'cf_ver' ], '>=' ) ) {
				$this->add_notice(  __( sprintf( $pattern, $processor[ 'name'], 'Caldera Forms', $processor[ 'cf_ver' ], CFCORE_VER  ) ) );
				$fail = true;
			}

		}

		if ( false != $processor[ 'php_ver' ] ) {
			if ( ! version_compare( PHP_VERSION, $processor[ 'php_ver' ], '>=' ) ) {
				$this->add_notice(  __( sprintf( $pattern, $processor[ 'name'], 'PHP', $processor[ 'php_ver' ], CFCORE_VER  ) ) );
				$fail = true;
			}

		}

		if ( false != $processor[ 'wp_ver' ] ) {
			global $wp_version;
			if ( ! version_compare( $wp_version, $processor[ 'wp_ver' ], '>=' ) ) {
				$this->add_notice( __( sprintf( $pattern, $processor[ 'name' ], 'WordPress ', $processor[ 'wp_ver' ], CFCORE_VER  ) ) );
				$fail = true;

			}

		}

		if ( isset( $processor[ 'icon' ] ) ) {
			$works = (@fopen($processor[ 'icon' ],"r")==true);
			if ( ! $works ) {
				unset( $processor[ 'icon' ] );
			}

		}

		if ( $fail ) {
			return false;

		}else{
			return $processor;

		}

	}

	/**
	 * Get class instance
	 *
	 * @since 1.3.0
	 *
	 * @return \Caldera_Forms_Processor_Load
	 */
	public static function get_instance() {
		if ( is_null( self::$init ) ) {
			self::$init = new self();
		}

		return self::$init;

	}

	/**
	 * Add an error notice in admin.
	 *
	 * @since 1.3
	 *
	 * @param string $notice The message.
 	 */
	protected function add_notice( $notice ) {
		if ( class_exists( 'Caldera_Forms_Admin' ) ) {
			Caldera_Forms_Admin::add_admin_notice( $notice );
		}

		$this->notices[] = $notice;
	}

}
